require('dotenv').config();
const express = require('express');
const cors = require('cors');
const { GoogleGenerativeAI } = require('@google/generative-ai');
const fs = require('fs').promises;
const path = require('path');

const app = express();
const port = process.env.PORT || 3001;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static('public')); // Serve static files if needed

// Initialize Gemini AI
const API_KEY = process.env.API_KEY || process.env.GENERATIVE_AI_API_KEY || '';
let genAI = null;
if (API_KEY) {
  try {
    genAI = new GoogleGenerativeAI(API_KEY);
  } catch (e) {
    console.error('Error initializing Generative AI client:', e.message || e);
  }
} else {
  console.warn('No API_KEY configured. Set API_KEY in environment or .env before starting the server.');
}

// Load company data
let companyData = '';

async function loadCompanyData() {
  try {
    const companyDataPath = path.join(__dirname, 'company-data.txt');
    companyData = await fs.readFile(companyDataPath, 'utf8');
    console.log('Company data loaded successfully');
    // Log a snippet of the data to confirm it's loaded
    console.log('Company data snippet:', companyData.substring(0, 100) + '...');
  } catch (error) {
    console.error('Error loading company data:', error);
  }
}

// Initialize company data on startup
async function startServer() {
  await loadCompanyData();
  
  app.listen(port, () => {
    console.log(`🚀 Marqenti Chatbot API server running on port ${port}`);
    console.log(`📊 Health check: http://localhost:${port}/api/health`);
    console.log(`💬 Chatbot endpoint: http://localhost:${port}/api/chat`);
    
    // Log environment status
    console.log(`🔑 API Key configured: ${!!API_KEY}`);
    console.log(`📄 Company data loaded: ${!!companyData && companyData.length > 0}`);
  });
}

startServer();

app.post('/api/chat', async (req, res) => {
  try {
    const { message, language = 'en' } = req.body;

    if (!message || !message.trim()) {
      return res.status(400).json({
        error: 'Message is required',
        response: 'Please provide a message.'
      });
    }

    if (!genAI) {
      return res.status(500).json({
        error: 'API key not configured',
        response: 'Sorry, the chatbot is currently unavailable. Please try again later.'
      });
    }

    // Language configuration can be kept as is, as it's generic
    const languageConfig = {
      'en': { name: 'English' }, 'de': { name: 'German (Deutsch)' }, 'fr': { name: 'French (Français)' },
      'es': { name: 'Spanish (Español)' }, 'ar': { name: 'Arabic (العربية)' }, 'it': { name: 'Italian (Italiano)' },
      'pt': { name: 'Portuguese (Português)' }, 'ru': { name: 'Russian (Русский)' }, 'tr': { name: 'Turkish (Türkçe)' },
      'zh': { name: 'Chinese (中文)' }
    };

    // Create the model and chat session
    const model = genAI.getGenerativeModel({ model: 'gemini-1.5-flash' });
    
    const chat = model.startChat({
      generationConfig: {
        maxOutputTokens: 800,
      },
    });

    // Create the enhanced prompt with automatic language detection
    const enhancedPrompt = `You are a helpful AI assistant for Marqenti, a creative marketing agency & digital/AI software developer.

IMPORTANT: Always respond in the SAME LANGUAGE that the user writes their message in. If they write in German, respond in German. If they write in English, respond in English. If they write in Arabic, respond in Arabic, etc. Automatically detect the user's language and match it exactly.

Be professional, helpful, and accurate. Always maintain a friendly and informative tone.

If asked about something not covered in the company data, politely explain that you can provide information specifically about Marqenti (in whatever language the user used).

Company Information:
${companyData}

---

User Question: ${message.trim()}

Please provide a helpful and informative response based on the company information above. Remember to automatically detect the user's language and respond in the EXACT SAME LANGUAGE they used.`;

    // Send message to Gemini
    const result = await chat.sendMessage(enhancedPrompt);
    const response = await result.response;
    const text = response.text();

    res.json({
      response: text,
      timestamp: new Date().toISOString()
    });

  } catch (error) {
    console.error('Chatbot error:', error);
    
    if (error.message && error.message.includes('API key not valid')) {
      return res.status(500).json({
        error: 'Invalid API key',
        response: 'Sorry, there\'s a configuration issue. Please contact support.'
      });
    }

    res.status(500).json({
      error: 'Internal server error',
      response: 'Sorry, I encountered an error while processing your request. Please try again in a moment.'
    });
  }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({
    status: 'OK',
    timestamp: new Date().toISOString(),
    companyDataLoaded: !!companyData
  });
});

// Get available languages endpoint
app.get('/api/languages', (req, res) => {
  const languages = {
    'en': { name: 'English', flag: '🇺🇸' }, 'de': { name: 'Deutsch', flag: '🇩🇪' },
    'fr': { name: 'Français', flag: '🇫🇷' }, 'es': { name: 'Español', flag: '🇪🇸' },
    'ar': { name: 'العربية', flag: '🇸🇦' }, 'it': { name: 'Italiano', flag: '🇮🇹' },
    'pt': { name: 'Português', flag: '🇵🇹' }, 'ru': { name: 'Русский', flag: '🇷🇺' },
    'tr': { name: 'Türkçe', flag: '🇹🇷' }, 'zh': { name: '中文', flag: '🇨🇳' }
  };
  
  res.json({
    languages: languages,
    defaultLanguage: 'en',
    total: Object.keys(languages).length
  });
});

// Get company info endpoint
app.get('/api/company-info', (req, res) => {
  res.json({
    name: 'Marqenti',
    description: 'A creative marketing agency & digital/AI software developer.',
    dataLoaded: !!companyData
  });
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Server error:', error);
  res.status(500).json({
    error: 'Internal server error',
    response: 'Sorry, something went wrong. Please try again later.'
  });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({
    error: 'Not found',
    response: 'The requested endpoint was not found.'
  });
});

module.exports = app;